<?php

/**
 * Contains the logic for deactivation popups
 * @since 1.0.0
 * @author woofunnels
 * @package WooFunnels
 */
#[AllowDynamicProperties]
class WooFunnels_Deactivate {

	public static $deactivation_str;

	/**
	 * Initialization of hooks where we prepare the functionality to ask use for survey
	 */
	public static function init() {


		add_action( 'admin_init', array( __CLASS__, 'load_all_str' ) );
		add_action( 'admin_footer', array( __CLASS__, 'maybe_load_deactivate_options' ) );

		add_action( 'wp_ajax_woofunnels_submit_uninstall_reason', array( __CLASS__, '_submit_uninstall_reason_action' ) );
	}

	/**
	 * Localizes all the string used
	 */
	public static function load_all_str() {

		self::$deactivation_str = array(
			'deactivation-share-reason'                => __( 'If you have a moment, please let us know why you are deactivating', 'woofunnels' ),
			'reason-no-longer-needed'                  => __( 'No longer need the plugin', 'woofunnels' ),
			'reason-found-a-better-plugin'             => __( 'I found an alternate plugin', 'woofunnels' ),
			'reason-needed-for-a-short-period'         => __( 'I only needed the plugin for a short period', 'woofunnels' ),
			'placeholder-plugin-name'                  => __( 'Please share which plugin', 'woofunnels' ),
			'reason-broke-my-site'                     => __( 'Encountered a fatal error', 'woofunnels' ),
			'reason-suddenly-stopped-working'          => __( 'The plugin suddenly stopped working', 'woofunnels' ),
			'reason-other'                             => _x( 'Other', 'the text of the "other" reason for deactivating the plugin that is shown in the modal box.', 'woofunnels' ),
			'deactivation-modal-button-submit'         => __( 'Submit & Deactivate', 'woofunnels' ),
			'deactivate'                               => __( 'Deactivate', 'woofunnels' ),
			'deactivation-modal-button-deactivate'     => __( 'Deactivate', 'woofunnels' ),
			'deactivation-modal-button-confirm'        => __( 'Yes - Deactivate', 'woofunnels' ),
			'deactivation-modal-button-cancel'         => _x( 'Cancel', 'the text of the cancel button of the plugin deactivation dialog box.', 'woofunnels' ),
			'reason-cant-pay-anymore'                  => __( "I can't pay for it anymore", 'woofunnels' ),
			'placeholder-comfortable-price'            => __( 'What price would you feel comfortable paying?', 'woofunnels' ),
			'reason-couldnt-make-it-work'              => __( "I couldn't understand how to make it work", 'woofunnels' ),
			'reason-great-but-need-specific-feature'   => __( "The plugin is great, but I need specific feature that you don't support", 'woofunnels' ),
			'reason-not-working'                       => __( 'Couldn\'t get the plugin to work', 'woofunnels' ),
			'reason-not-what-i-was-looking-for'        => __( "It's not what I was looking for", 'woofunnels' ),
			'reason-didnt-work-as-expected'            => __( "The plugin didn't work as expected", 'woofunnels' ),
			'placeholder-feature'                      => __( 'What feature?', 'woofunnels' ),
			'placeholder-share-what-didnt-work'        => __( "Kindly share what didn't work so we can fix it for future users...", 'woofunnels' ),
			'placeholder-what-youve-been-looking-for'  => __( "What you've been looking for?", 'woofunnels' ),
			'placeholder-what-did-you-expect'          => __( 'What did you expect?', 'woofunnels' ),
			'reason-didnt-work'                        => __( "The plugin didn't work", 'woofunnels' ),
			'reason-dont-like-to-share-my-information' => __( "I don't like to share my information with you", 'woofunnels' ),
			'conflicts-other-plugins'                  => __( "Conflicts with other plugins", 'woofunnels' ),
			'temporary-deactivation'                   => __( "It's a temporary deactivation", 'woofunnels' ),
		);
	}

	/**
	 * Checking current page and pushing html, js and css for this task
	 * @global string $pagenow current admin page
	 * @global array $VARS global vars to pass to view file
	 */
	public static function maybe_load_deactivate_options() {
		global $pagenow;

		if ( $pagenow === 'plugins.php' ) {
			global $VARS;

			$VARS = array(
				'slug'    => '',
				'reasons' => self::deactivate_options(),
			);

			include_once dirname( dirname( __FILE__ ) ) . '/views/woofunnels-deactivate-modal.phtml';
		}
	}

	/**
	 * deactivation reasons in array format
	 * @return array reasons array
	 * @since 1.0.0
	 */
	public static function deactivate_options() {

		$reasons            = array( 'default' => [] );
		$reasons['default'] = array(
			array(
				'id'                => 1,
				'text'              => self::load_str( 'reason-not-working' ),
				'input_type'        => '',
				'input_placeholder' => self::load_str( 'placeholder-plugin-name' ),
			),
			array(
				'id'                => 2,
				'text'              => self::load_str( 'reason-found-a-better-plugin' ),
				'input_type'        => 'textfield',
				'input_placeholder' => self::load_str( 'placeholder-plugin-name' )
			),
			array(
				'id'                => 3,
				'text'              => self::load_str( 'reason-no-longer-needed' ),
				'input_type'        => '',
				'input_placeholder' => '',
			),

			array(
				'id'                => 4,
				'text'              => self::load_str( 'temporary-deactivation' ),
				'input_type'        => '',
				'input_placeholder' => '',
			),
			array(
				'id'                => 5,
				'text'              => self::load_str( 'conflicts-other-plugins' ),
				'input_type'        => 'textfield',
				'input_placeholder' => self::load_str( 'placeholder-plugin-name' )
			),
			array(
				'id'                => 6,
				'text'              => self::load_str( 'reason-broke-my-site' ),
				'input_type'        => '',
				'input_placeholder' => '',
			),
			array(
				'id'                => 7,
				'text'              => self::load_str( 'reason-other' ),
				'input_type'        => 'textfield',
				'input_placeholder' => __( 'Please share the reason', 'woofunnels' ),
			)
		);


		return $reasons;
	}

	/**
	 * get exact str against the slug
	 *
	 * @param $slug
	 *
	 * @return mixed
	 */
	public static function load_str( $slug ) {
		return self::$deactivation_str[ $slug ];
	}

	/**
	 * Called after the user has submitted his reason for deactivating the plugin.
	 *
	 * @since  1.1.2
	 */
	public static function _submit_uninstall_reason_action() {
		check_admin_referer( 'bwf_secure_key', '_nonce' );

		if ( ! current_user_can( 'install_plugins' ) ) {
			wp_send_json_error();
		}
		if ( ! isset( $_POST['reason_id'] ) ) {
			exit;
		}

		$reason_info = isset( $_POST['reason_info'] ) ? sanitize_textarea_field( stripslashes( bwf_clean( $_POST['reason_info'] ) ) ) : '';

		$reason = array(
			'id'   => sanitize_text_field( $_POST['reason_id'] ),
			'info' => substr( $reason_info, 0, 128 ),
		);

		$licenses        = WooFunnels_addons::get_installed_plugins();
		$version         = 'NA';
		$plugin_basename = isset( $_POST['plugin_basename'] ) ? bwf_clean( $_POST['plugin_basename'] ) : '';

		if ( $licenses && count( $licenses ) > 0 ) {
			foreach ( $licenses as $key => $license ) {
				if ( $key === $plugin_basename ) {
					$version = $license['Version'];
				}
			}
		}

		$deactivations = array(
			$plugin_basename . '(' . $version . ')' => $reason,
		);


		WooFunnels_API::post_deactivation_data( $deactivations );
		wp_send_json_success();

	}

	public static function get_recent_fatal_error_logs( $limit = 2 ) {
		try {
			// Get the WordPress uploads directory information
			$upload_dir          = wp_upload_dir();
			$log_directory       = trailingslashit( $upload_dir['basedir'] ) . 'wc-logs/';
			$log_url_base        = trailingslashit( $upload_dir['baseurl'] ) . 'wc-logs/';
			$fatal_error_pattern = '/^fatal-errors-.*\.log$/';
			$log_files           = [];

			// Check if the directory exists
			if ( ! is_dir( $log_directory ) ) {
				return [];
			}

			// Scan the directory for files
			$files = scandir( $log_directory );
			if ( $files === false ) {
				throw new Exception( "Failed to scan directory: $log_directory" );
			}

			foreach ( $files as $file ) {
				if ( preg_match( $fatal_error_pattern, $file ) ) {
					$full_path = $log_directory . $file;
					$mtime     = filemtime( $full_path );
					if ( $mtime === false ) {
						throw new Exception( "Failed to get modification time for file: $full_path" );
					}
					$log_files[ $file ] = $mtime;
				}
			}

			// Sort files by modification time, newest first
			arsort( $log_files );

			// Get the full URLs of the most recent log files
			$recent_log_urls = [];
			$count           = 0;
			foreach ( $log_files as $file => $mtime ) {
				if ( $count >= $limit ) {
					break;
				}
				$recent_log_urls[] = $log_url_base . $file;
				$count ++;
			}

			return $recent_log_urls;
		} catch ( Exception|Error $e ) {
			// Log the exception or error
			return [];
		}
	}

}
